﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Infrastructure.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// <inheritdoc />
    [Authorize]
    [Route("api/salutation")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class SalutationController : BaseController
    {
        /// <summary>
        /// The salutation services.
        /// </summary>
        private readonly ISalutationService salutationService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public SalutationController(ISalutationService salutationService, IAuditLogService auditLogService)
        {
            this.salutationService = salutationService;
            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] SalutationModel model, [FromHeader] LocationHeader header)
        {
            model = (SalutationModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);

            var response = await this.salutationService.InsertAsync(model);

            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Salutation,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has added <b>Salutation</b> of <strong>{model.Name}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        /// <summary>
        /// update the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] SalutationModel model, [FromHeader] LocationHeader header)
        {
            model = (SalutationModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);

            var response = await this.salutationService.UpdateAsync(model);

            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Salutation,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has updated <b>Salutation</b> of <strong>{model.Name}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] SalutationModel model)
        {
            model = (SalutationModel)EmptyFilter.Handler(model);
            var response = await this.salutationService.FetchAllAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-active")]
        public async Task<ActionResult> FetchActiveAsync([FromBody] SalutationModel model)
        {
            model = (SalutationModel)EmptyFilter.Handler(model);
            var response = await this.salutationService.FetchActiveAllAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// modify asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] SalutationModel model, [FromHeader] LocationHeader header)
        {
            model = (SalutationModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);
            var response = await this.salutationService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Salutation,
                LogFrom = short.Parse(model.LoginRoleId.ToString()),
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LocationId = (int)model.LocationId,
                LogDescription = $@"<b>{model.CreatedByName}</b> has <b>{(model.Active ? "Activated" : "Deactivated")}</b> the Salutation <b>{model.Name}</b>  successfully"
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }

    }
}

